﻿using System;
using System.Linq;
using System.Collections.Generic;

using AutoMapper;

using HIPS.PcehrDataStore.Schemas;
using HIPS.Web.UI.ViewModels.PcehrView;

namespace HIPS.Web.UI.Helpers.Mapping.Profiles
{

    /// <summary>
    /// AutoMapper mapping profile for the Hospital entity.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
    /// </history>
    internal class HospitalProfile : Profile
    {

        #region Declarations

        #region Resolvers

        /// <summary>
        /// Resolves the value of the HospitalCode based on a provided hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        internal abstract class HospitalCodeResolverBase : IValueResolver
        {

            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
            /// </history>
            public virtual ResolutionResult Resolve(ResolutionResult source)
            {

                if (source == null)
                {
                    throw new System.ArgumentNullException("source");
                }

                HIPS.PcehrDataStore.Schemas.HospitalCode hospitalCode;

                var context = source.Context.Options.Items.Values.OfType<Context.HospitalMappingContext>().SingleOrDefault();

                if (! string.IsNullOrEmpty(context.HospitalCodeSystem))
                {
                    //Use hospital code system provided in context.
                    hospitalCode = (source.Value as HIPS.PcehrDataStore.Schemas.Hospital).Codes.SingleOrDefault(c => c.CodeSystemCode == context.HospitalCodeSystem);
                }
                else
                {
                    //Use first code as fallback.
                    hospitalCode = (source.Value as HIPS.PcehrDataStore.Schemas.Hospital).Codes[0];
                }

                return source.New(hospitalCode);

            }

            #endregion

        }

        /// <summary>
        /// Resolves the value of the HospitalCodeSystem based on a provided hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        internal class HospitalCodeSystemResolver : HospitalCodeResolverBase
        {

            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
            /// </history>
            public override ResolutionResult Resolve(ResolutionResult source)
            {
                return source.New((base.Resolve(source).Value as HIPS.PcehrDataStore.Schemas.HospitalCode).CodeSystemCode);
            }

            #endregion

        }

        /// <summary>
        /// Resolves the value of the HospitalCode based on a provided hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        internal class HospitalCodeResolver : HospitalCodeResolverBase
        {

            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
            /// </history>
            public override ResolutionResult Resolve(ResolutionResult source)
            {
                return source.New((base.Resolve(source).Value as HIPS.PcehrDataStore.Schemas.HospitalCode).Code);
            }

            #endregion

        }

        #endregion

        #endregion

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        protected override void Configure()
        {
            //Data --> ViewModel:
            Mapper.CreateMap<Hospital, HospitalViewModel>()
                .ForMember(dest => dest.CodeSystemCode, opt => opt.ResolveUsing<HospitalCodeSystemResolver>())
                .ForMember(dest => dest.Code, opt => opt.ResolveUsing<HospitalCodeResolver>()) 
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.Name))
                ;
        }

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion

    }
}
